#############################################################################
#############################################################################
#
# latexMacros.tcl (called from latex.tcl)
#
#############################################################################
#
# Author:  Tom Scavo <trscavo@syr.edu>, Vince Darley <vince@santafe.edu>
#
#############################################################################
#############################################################################
proc latexMacros.tcl {} {}
#############################################################################
# 
# Basic Commands
#
#############################################################################
#  label returners  #

proc TeX::labelDelim {} {
    global TeXmodeVars
    return $TeXmodeVars(standardTeXLabelDelimiter)
}

proc TeX::label {type} {
    global TeXmodeVars
    return "\\label\{${type}$TeXmodeVars(standardTeXLabelDelimiter)\}"
}

#--------------------------------------------------------------------------
#  Utilities: 
#--------------------------------------------------------------------------
# A keyboard-bound method of accessing menu commands.  Takes a list of 
# menu items (i.e., the tail of a 'menu' command), the menu name (the 
# argument of the '-n' switch) , and the name of a menu filter (the 
# argument of the '-p' switch) as input, and displays these items in a 
# list box.  If the chosen item is a menu command (as opposed to a 
# submenu), it is passed to the menu filter; otherwise, 'chooseCommand' 
# recursively calls itself until a menu command is chosen or the cancel 
# button is pressed.
#

proc chooseCommand {menuItems {menuName ""} {menuFilterProc ""} {level 1}} {
    if {![string length $menuItems]} {return}
    watchCursor
    # Preprocess the list of menu items:
    foreach item $menuItems {
	regsub -all {[<!/].} $item {} item
	regsub -all {}	$item {} item
	lappend	menOut $item
	if {[string match "menu*" $item]} {
	    if {[set ind [lsearch $item {-n}]] >= 0} {
		lappend	top "[lindex $item [incr ind]]:"
	    }
	} elseif {![string match "(*" $item]} {
	    lappend top $item
	}
    }
    # Present the menu items to the user:
    set res [listpick -p "Choose menu command (level $level):" $top]
    # Either execute a command or recurse on a submenu:
    if {[lsearch $menOut $res] >= 0}  {
	# Execute the command via the menu filter, if necessary:
	if {$menuFilterProc == ""} {
	    $res
	} else {
	    $menuFilterProc $menuName $res
	}
    } else {
	set res [string trimright $res {:}]
	foreach	item $menOut {
	    if {[lsearch $item $res] >= 0} {
		set menuItems [lindex $item end]
		# Determine the name of this submenu:
		if {[set ind [lsearch $item {-n}]] >= 0} {
		    set menuName [lindex $item [incr ind]]
		} else {
		    set menuName ""
		}
		# Determine the name of the menu filter for this submenu:
		if {[set ind [lsearch $item {-p}]] >= 0} {
		    set menuFilterProc [lindex $item [incr ind]]
		} else {
		    set menuFilterProc ""
		}
		return [chooseCommand $menuItems $menuName $menuFilterProc [incr level]]
	    }
	}
    }
}

# A keyboard-bound method of inserting environments.
#
proc chooseEnvironment {} {
    set env [getEnvironment]
    if {[catch {eval $env}]} {doWrapEnvironment $env}
}

proc getEnvironment {} {
    global useAMSLaTeX
    if { $useAMSLaTeX } {
	set environments [list \
	  "abstract" \
	  "align" \
	  "alignat" \
	  "aligned" \
	  "array" \
	  "bmatrix" \
	  "Bmatrix" \
	  "cases" \
	  "center" \
	  "corollary" \
	  "definition" \
	  "description" \
	  "displaymath" \
	  "document" \
	  "enumerate" \
	  "eqnarray" \
	  "equation" \
	  "figure" \
	  "filecontents" \
	  "flushleft" \
	  "flushright" \
	  "gather" \
	  "gathered" \
	  "itemize" \
	  "lemma" \
	  "matrix" \
	  "math" \
	  "minipage" \
	  "multline" \
	  "note" \
	  "overlay" \
	  "pmatrix" \
	  "proof" \
	  "proposition" \
	  "quotation" \
	  "quote" \
	  "remark" \
	  "slide" \
	  "smallmatrix" \
	  "split" \
	  "subarray" \
	  "subequations" \
	  "table" \
	  "tabular" \
	  "thebibliography" \
	  "theorem" \
	  "titlepage" \
	  "verbatim" \
	  "verse" \
	  "vmatrix" \
	  "Vmatrix" ]
    } else {
	set environments [list \
	  "abstract" \
	  "array" \
	  "center" \
	  "description" \
	  "displaymath" \
	  "document" \
	  "enumerate" \
	  "eqnarray" \
	  "equation" \
	  "figure" \
	  "filecontents" \
	  "flushleft" \
	  "flushright" \
	  "itemize" \
	  "math" \
	  "minipage" \
	  "note" \
	  "overlay" \
	  "quotation" \
	  "quote" \
	  "slide" \
	  "table" \
	  "tabular" \
	  "thebibliography" \
	  "titlepage" \
	  "verbatim" \
	  "verse" ]
    }
    set p "Choose environment:"
    if {[catch {eval [list prompt $p abstract ""] $environments} env]} {
	set env ""
    }
    return $env
}

#############################################################################
# 
# Paragraph Mode Macros
#
#############################################################################
#--------------------------------------------------------------------------
#  Documents: 
#--------------------------------------------------------------------------

proc newLaTeXDocument {} {
    set classes [list "article" "report" "book" "letter" "slides"]
    set p {"Choose a documentclass:"}
    if {![catch {eval prompt $p "article" "classes:" $classes} documentType]} {
	new -m TeX
	if {[catch {${documentType}Documentclass}]} {
	    wrapDocument "$documentType" 
	}
	while {[options]} {
	    status::msg "enter option (or leave blank)"
	}
    }
    status::msg ""
}

proc letterDocumentclass {} {
    set    preamble "\r\\address\{%\r"
    append preamble "		\\\\	% insert your name here\r"
    append preamble "		\\\\	% insert your address here\r"
    append preamble "		\\\\	% insert more address here\r"
    append preamble "		  	% insert city-state-zip here\r"
    append preamble "\}\r\r"
    append preamble "\\date\{\}  % optional\r"
    append preamble "\\signature\{\}\r\r"
    set    body     "\r\\begin\{letter\}\{%\r"
    append body     "		\\\\	% insert addressee's name here\r"
    append body     "		\\\\	% insert addressee's address here\r"
    append body     "		\\\\	% insert more address here\r"
    append body     "		  	% insert addressee's city-state-zip here\r"
    append body     "\}\r\r"
    append body     "\\opening\{Dear ,\}\r\r"
    if {[isEmptyFile]} {
	append body "% BODY OF LETTER\r"
	append body "\r\r"
    } else {
	if {[isSelectionAll]} {
	    set text [getSelect]
	    # deleteText [minPos] [maxPos]
	    append body "$text\r"
	} else {
	    alertnote "nonempty file:  delete text or \'Select All\'\
	      from the Edit menu"
	    return
	}
    }
    append body "\\closing\{Sincerely,\}\r\r"
    append body "\\encl\{\}\r"
    append body "\\cc\{\}\r\r"
    append body "\\end\{letter\}\r\r"
    insertDocument "letter" $preamble $body
    status::msg "enter option (or leave blank)"
}

proc articleDocumentclass {} {
    if {[wrapDocument "article"]} {
	status::msg "enter option (or leave blank)"
    }
}
proc reportDocumentclass {} {
    if {[wrapDocument "report"]} {
	status::msg "enter option (or leave blank)"
    }
}
proc bookDocumentclass {} {
    if {[wrapDocument "book"]} {
	status::msg "enter option (or leave blank)"
    }
}
proc slidesDocumentclass {} {
    if {[wrapDocument "slides"]} {
	status::msg "enter option (or leave blank)"
    }
}
proc otherDocumentclass {} {
    catch {prompt "What documentclass?" "article"} documentType
    if {$documentType != "cancel"} {
	if {[wrapDocument "$documentType"]} {
	    status::msg "enter option (or leave blank)"
	}
    }
}
# If an option is inserted, return true; otherwise, return false.
proc options {} {
    set option [getOption]
    if {$option != "" && $option != "cancel"} {
	insertOption $option
	return 1
    }
    return 0
}
proc getOption {} {
    set options [list \
      "10pt" "11pt" "12pt" "(-" \
      "letterpaper" "legalpaper" "executivepaper" "a4paper" "a5paper" \
      "b5paper" "(-" "landscape" "(-" "final" "draft" "(-" \
      "oneside" "twoside" "(-" "openright" "openany" "(-" \
      "onecolumn" "twocolumn" "(-" "notitlepage" "titlepage" \
      "openbib" "(-" "leqno" "(-" "fleqn"] 
    set p {"Choose an option:"}
    if {![catch {eval prompt $p "11pt" options: $options} option]} {
	return $option
    } else {
	return ""
    }
}
proc insertOption {option} {
    global TeXmodeVars
    set searchString {\\documentclass}
    set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString [minPos]]
    if {[llength $searchResult] == 0} {
	if { $TeXmodeVars(searchNoisily) } {beep}
	status::msg "can\'t find \\documentclass"
    } else {
	set nextCharPos [lindex $searchResult 1]
	goto $nextCharPos
	set nextChar [lookAt $nextCharPos]
	if {$nextChar == "\["} {
	    forwardChar
	    insertText $option
	    if {[lookAt [getPos]] != "\]"} {
		insertText ","
	    }
	} elseif {$nextChar == "\{"} {
	    insertText "\[$option\]"
	} else {
	    alertnote "unrecognizable \\documentclass statement"
	}
    }
}
proc insertPackage {package} {
    global TeXmodeVars
    # Check to see if $package is already loaded:
    if {$package != ""} {
	append searchString {^[^%]*\\usepackage\{.*} $package {.*\}}
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString [minPos]]
	if {[llength $searchResult] != 0} {
	    if { $TeXmodeVars(searchNoisily) } {beep}
	    status::msg "$package package already loaded"
	    return
	}
    }
    # Newlines are allowed in the arguments of \documentclass:
    set searchString {\\documentclass(\[[^][]*\])?{[^{}]*}}
    # Search for \documentclass command:
    set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString [minPos]]
    if {[llength $searchResult] == 0} {
	if { $TeXmodeVars(searchNoisily) } {beep}
	status::msg "can't find \\documentclass"
    } else {
	pushPosition
	goto [lindex $searchResult 1]
	set txt "\r\\usepackage\{$package\}"
	insertText $txt
	backwardChar
	status::msg "Press <Ctl .> to return to previous position"
    }
}
proc filecontents {} {
    global searchNoisily
    set searchString {\\documentclass}
    set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString [minPos]]
    if {[llength $searchResult] == 0} {
	if {$searchNoisily} {beep}
	status::msg "can\'t find \\documentclass"
	return
    } else {
	set prompt "File to be included:"
	if {[catch {getfile $prompt} path]} {
	    return
	} else {
	    replaceText [minPos] [minPos] [buildFilecontents $path]
	    goto [minPos]
	    status::msg "file included"
	}
    }
}
proc filecontentsAll {} {
    global searchNoisily
    watchCursor
    status::msg "locating all input files"
    set currentWin [win::Current]
    # Is the current window part of TeX fileset?
    set fset [isWindowInFileset $currentWin "tex"]
    if { $fset == "" } {
	set searchString {\\documentclass}
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString [minPos]]
	if {[llength $searchResult] == 0} {
	    if {$searchNoisily} {beep}
	    status::msg "can\'t find \\documentclass"
	    return
	} else {
	    set text [getText [minPos] [maxPos]]
	}
    } else {
	# Will not handle a base file that is open and dirty:
	set text [buildFilecontents [texFilesetBaseName $fset]]
    }
    set currentDir [file dirname $currentWin]
    set newText [texResolveAll $text $currentDir]
    if {[string length $text] == [string length $newText]} {
	beep
	status::msg "no files to include"
    } else {
	replaceText [minPos] [maxPos] $newText
	goto [minPos]
	status::msg "all files included"
    }
}
# Takes a LaTeX document string and a path as input, and returns
# a modified document string with all filecontents environments
# prepended.
proc texResolveAll {latexDoc currentDir} {
    global TeXmodeVars
    set pairs [list \
      {{\\documentclass} {.cls}} {{\\LoadClass} {.cls}} \
      {{\\include} {.tex}} \
      {{\\usepackage} {.sty}} {{\\RequirePackage} {.sty}} \
      {{\\input} {}} \
      {{\\bibliography} {.bib}} {{\\bibliographystyle} {.bst}} \
      ]
    foreach macro $TeXmodeVars(boxMacroNames) {
	regsub {\*} $macro {\\*} macro
	lappend pairs [list \\\\$macro {}]
    }
    foreach pair $pairs {
	set cmd [lindex $pair 0]
	set ext [lindex $pair 1]
	set searchString $cmd
	append searchString {(\[[^][]*\])?{([^{}]*)}}
	set searchText $latexDoc
	while {[regexp -indices -- $searchString $searchText mtch dummy theArgs]} {
	    set begPos [lindex $theArgs 0]
	    set endPos [lindex $theArgs 1]
	    set args [string range $searchText $begPos $endPos]
	    foreach arg [split $args ,] {
		if { $cmd == {\\input} && ![string length [file extension $arg]]} {
		    set ext {.tex}
		}
		set files [glob -nocomplain -path [file join $currentDir $arg] *]
		set filename [file join $currentDir $arg$ext]
		if {[lsearch -exact $files $filename] > -1 } {
		    set tempDoc $latexDoc
		    set latexDoc [buildFilecontents $filename]
		    append latexDoc $tempDoc
		}
	    }
	    set searchText [string range $searchText [expr $endPos + 2] end]
	}
    }
    return $latexDoc
}
# Takes a filename as input and returns a filecontents environment 
# based on the contents of that file.  If a second argument is given,
# use that as the argument of the filecontents environment instead
# of the original filename.
proc buildFilecontents {filename {newFilename {}}} {
    set text [file::readAll $filename]
    # Fix end-of-line characters:
    regsub -all "\xa" $text "\xd" text
    set envName "filecontents"
    if { $newFilename == {} } {
	set envArg "{[file tail $filename]}"
    } else {
	set envArg "{$newFilename}"
    }
    return [buildEnvironment $envName $envArg "$text\r" "\r\r"]
}
#--------------------------------------------------------------------------
#  Page Layout: 
#--------------------------------------------------------------------------
proc maketitle {} {
    global searchNoisily
    set searchString {\\document(class|style)(\[.*\])?\{.*\}}
    set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString [minPos]]
    if {[llength $searchResult] == 0} {
	if {$searchNoisily} {beep}
	status::msg "can\'t find \\documentclass or \\documentstyle"
    } else {
	set searchPos [lindex $searchResult 1]
	set searchString {\\begin\{document\}}
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString $searchPos]
	if {[llength $searchResult] == 0} {
	    if {$searchNoisily} {beep}
	    status::msg "can\'t find \\begin\{document\}"
	} else {
	    goto [lindex $searchResult 1]
	    set currentPos [getPos]
	    set txt "\r\r% Definition of title page:"
	    append txt "\r\\title\{"
	    append txt "\r\ttitle\r\}"
	    append txt "\r\\author\{"
	    append txt "\r\t\t% insert author(s) here"
	    append txt "\r\}"
	    append txt "\r\\date\{\}\t% optional"
	    append txt "\r\r\\maketitle"
	    elec::Insertion $txt
	}
    }
}
proc abstract {} { doWrapEnvironment "abstract" }
proc titlepage {} { doWrapEnvironment "titlepage" }
proc getPagestyle {} {
    set styles [list "plain" "empty" "headings" "myheadings"]
    set p {"Choose a pagestyle:"}
    if {![catch {eval prompt $p "plain" "options:" $styles} pagestyleName]} {
	return $pagestyleName
    } else {
	return ""
    }
}
proc pagestyle {} {
    set pagestyleName [getPagestyle]
    if {$pagestyleName != ""} {
	insertObject "[openingCarriageReturn]\\pagestyle\{$pagestyleName\}[closingCarriageReturn]"
    }
}
proc thispagestyle {} {
    set pagestyleName [getPagestyle]
    if {$pagestyleName != ""} {
	insertObject "[openingCarriageReturn]\\thispagestyle\{$pagestyleName\}[closingCarriageReturn]"
    }
}

proc pagenumbering {} {
    set pagenumberingStyle [getPagenumberingStyle]
    if {$pagenumberingStyle != ""} {
	insertObject "[openingCarriageReturn]\\pagenumbering\{$pagenumberingStyle\}[closingCarriageReturn]"
    }
}
proc getPagenumberingStyle {} {
    set styles [list "arabic" "roman" "Roman" "alph" "Alph"]
    set p {"Choose a pagenumbering style:"}
    if {![catch {eval prompt $p "arabic" "options:" $styles} pagenumberingStyle]} {
	return $pagenumberingStyle
    } else {
	return ""
    }
}
proc twocolumn {} {
    insertObject "[openingCarriageReturn]\\twocolumn[closingCarriageReturn]"
}
proc onecolumn {} {
    insertObject "[openingCarriageReturn]\\onecolumn[closingCarriageReturn]"
}

#--------------------------------------------------------------------------
#  Sectioning: 
#--------------------------------------------------------------------------

proc sectioning {item} {
    append left [openingCarriageReturn] "\\${item}\{"
    append right "\}"
    if [regexp (part|chapter) $item] {
	append right "\\thispagestyle{empty}"
    }
    append right "" [closingCarriageReturn]
    if [elec::Wrap $left $right] {
	status::msg "don't forget the label <Ctl Opt L>"
    } else {
	status::msg "type the [string trim $item {*}] name and don't forget the label <Ctl Opt L>"
    }
}
proc appendix {} {
	insertObject "[openingCarriageReturn]\\appendix[closingCarriageReturn]"
}

#--------------------------------------------------------------------------
#  Text Style: 
#--------------------------------------------------------------------------

proc emph {} {
    if {[elec::Wrap "\\emph{" "}"]} {
	status::msg "selected text has been emphasized"
    } else {
	status::msg "enter text to be emphasized"
    }
}
proc textup {} {
    if {[elec::Wrap "\\textup{" "}"]} {
	status::msg "selected text has upright shape"
    } else {
	status::msg "enter text to have upright shape"
    }
}
proc textit {} {
    if {[elec::Wrap "\\textit{" "}"]} {
	status::msg "selected text has italic shape"
    } else {
	status::msg "enter text to have italic shape"
    }
}
proc textsl {} {
    if {[elec::Wrap "\\textsl{" "}"]} {
	status::msg "selected text has slanted shape"
    } else {
	status::msg "enter text to have slanted shape"
    }
}
proc textsc {} {
    if {[elec::Wrap "\\textsc{" "}"]} {
	status::msg "selected text has small caps shape"
    } else {
	status::msg "enter text to have small caps shape"
    }
}
proc textmd {} {
    if {[elec::Wrap "\\textmd{" "}"]} {
	status::msg "selected text has been set in medium series"
    } else {
	status::msg "enter text to be set in medium series"
    }
}
proc textbf {} {
    if {[elec::Wrap "\\textbf{" "}"]} {
	status::msg "selected text has been set in bold series"
    } else {
	status::msg "enter text to be set in bold series"
    }
}
proc textrm {} {
    if {[elec::Wrap "\\textrm{" "}"]} {
	status::msg "selected text has been set with roman family"
    } else {
	status::msg "enter text to be set using roman family"
    }
}
proc textsf {} {
    if {[elec::Wrap "\\textsf{" "}"]} {
	status::msg "selected text has been set with sans serif family"
    } else {
	status::msg "enter text to be set using sans serif family"
    }
}
proc texttt {} {
    if {[elec::Wrap "\\texttt{" "}"]} {
	status::msg "selected text has been set with typewriter family"
    } else {
	status::msg "enter text to be set using typewriter family"
    }
}
proc textnormal {} {
    if {[elec::Wrap "\\textnormal{" "}"]} {
	status::msg "selected text has been set with normal style"
    } else {
	status::msg "enter text to be set using normal style"
    }
}

proc em {} {
    if {[elec::Wrap "{\\em " "}"]} {
	status::msg "emphasized text set"
    } else {
	status::msg "enter text to be emphasized"
    }
}
proc upshape {} {
    if {[elec::Wrap "{\\upshape " "}"]} {
	status::msg "text set in upright shape"
    } else {
	status::msg "enter text to be set in upright shape"
    }
}
proc itshape {} {
    if {[elec::Wrap "{\\itshape " "}"]} {
	status::msg "text set in italics shape"
    } else {
	status::msg "enter text to be set in italics shape"
    }
}
proc slshape {} {
    if {[elec::Wrap "{\\slshape " "}"]} {
	status::msg "text set in slanted shape"
    } else {
	status::msg "enter text to be set in slanted shape"
    }
}
proc scshape {} {
    if {[elec::Wrap "{\\scshape " "}"]} {
	status::msg "text set in small caps shape"
    } else {
	status::msg "enter text to be set in small caps shape"
    }
}
proc mdseries {} {
    if {[elec::Wrap "{\\mdseries " "}"]} {
	status::msg "text set in medium series"
    } else {
	status::msg "enter text to be set in medium series"
    }
}
proc bfseries {} {
    if {[elec::Wrap "{\\bfseries " "}"]} {
	status::msg "text set in bold series"
    } else {
	status::msg "enter text to be set in bold series"
    }
}
proc rmfamily {} {
    if {[elec::Wrap "{\\rmfamily " "}"]} {
	status::msg "text set in roman family"
    } else {
	status::msg "enter text to be set in roman family"
    }
}
proc sffamily {} {
    if {[elec::Wrap "{\\sffamily " "}"]} {
	status::msg "text set in sans serif family"
    } else {
	status::msg "enter text to be set in sans serif family"
    }
}
proc ttfamily {} {
    if {[elec::Wrap "{\\ttfamily " "}"]} {
	status::msg "text set in typewriter family"
    } else {
	status::msg "enter text to be set in typewriter family"
    }
}
proc normalfont {} {
    if {[elec::Wrap "{\\normalfont " "}"]} {
	status::msg "text set in normal style"
    } else {
	status::msg "enter text to be set in normal style"
    }
}

#--------------------------------------------------------------------------
#  Text Size: 
#--------------------------------------------------------------------------

proc doTextSize {textSize} {
    if {[elec::Wrap "{\\$textSize " "}"]} {
	status::msg "$textSize text set"
    } else {
	status::msg "enter $textSize text"
    }
}

#--------------------------------------------------------------------------
#  Text Commands: 
#--------------------------------------------------------------------------

proc textsuperscript {} {
    if {[elec::Wrap "\\textsuperscript{" "}"]} {
	status::msg "text superscripted"
    } else {
	status::msg "enter superscripted text"
    }
}
proc textcircled {} {
    if {[elec::Wrap "\\textcircled{" "}"]} {
	status::msg "text circled"
    } else {
	status::msg "enter circled text"
    }
}

#--------------------------------------------------------------------------
#  International: 
#--------------------------------------------------------------------------

proc {} {} {
    if {[elec::Wrap "\\`{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter single character"
    }
}
proc {} {} {
    if {[elec::Wrap "\\'{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter single character"
    }
}
proc {} {} {
    if {[elec::Wrap "\\^{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter single character"
    }
}
proc {} {} {
    if {[elec::Wrap "\\\"{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter single character"
    }
}
proc {} {} {
    if {[elec::Wrap "\\~{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter single character"
    }
}
proc {} {} {insertObject "\\c\{c\}"}
proc {} {} {insertObject "\\c\{C\}"}
proc {} {} {insertObject "\\oe"}
proc {} {} {insertObject "\\OE"}
proc {} {} {insertObject "\\ae"}
proc {} {} {insertObject "\\AE"}
proc {} {} {insertObject "\\aa"}
proc {} {} {insertObject "\\AA"}
proc {} {} {insertObject "\\o"}
proc {} {} {insertObject "\\O"}
proc {ss} {} {insertObject "\\ss"}
proc {SS} {} {insertObject "\\SS"}
proc {} {} {insertObject "?`"}
proc {} {} {insertObject "!`"}


#--------------------------------------------------------------------------
#  Boxes: 
#--------------------------------------------------------------------------

proc mbox {} {
    if {[elec::Wrap "\\mbox{" "}"]} {
	status::msg "mbox set"
    } else {
	status::msg "enter text"
    }
}
proc makebox {} {
    if {[elec::Wrap "\\makebox\[\]\[\]{" "}"]} {
	status::msg "makebox set; enter the width and position"
    } else {
	status::msg "enter the width and position of the makebox, then the text"
    }
}
proc fbox {} {
    if {[elec::Wrap "\\fbox{" "}"]} {
	status::msg "fbox set"
    } else {
	status::msg "enter text"
    }
}
proc framebox {} {
    if {[elec::Wrap "\\framebox\[\]\[\]{" "}"]} {
	status::msg "framebox set; enter the width and position"
    } else {
	status::msg "enter the width and position of the framebox, then the text"
    }
}
proc newsavebox {} {
    if {[elec::Wrap "\\newsavebox{" "}"]} {
	status::msg "newsavebox defined"
    } else {
	status::msg "enter the command name of the sbox or savebox"
    }
}
proc sbox {} {
    if {[elec::Wrap "\\sbox{}{" "}"]} {
	status::msg "sbox set; enter the command name"
    } else {
	status::msg "enter the command name of the sbox, then the text"
    }
}
proc savebox {} {
    if {[elec::Wrap "\\savebox{}\[\]\[\]{" "}"]} {
	status::msg "savebox set; enter the command name"
    } else {
	status::msg "enter the command name of the savebox"
    }
}
proc usebox {} {
    if {[elec::Wrap "\\usebox{" "}"]} {
	status::msg "usebox declared"
    } else {
	status::msg "enter the command name of the sbox or savebox"
    }
}
proc raisebox {} {
    if {[elec::Wrap "\\raisebox{}\[\]\[\]{" "}"]} {
	status::msg "raisebox set; enter the displacement"
    } else {
	status::msg "enter the displacement of the raisebox"
    }
}
proc parbox {} {
    if {[elec::Wrap "\\parbox\[\]\{\}{" "}"]} {
	status::msg "parbox set; enter the position and width"
    } else {
	status::msg "enter the position \[b|c|t\] and width of the parbox, then the text"
    }
}
proc rule {} {
    insertObject "\\rule\[\]\{\}{}"
    status::msg "enter the displacement of the rule, then width and height"
}

#--------------------------------------------------------------------------
#  Misc: 
#--------------------------------------------------------------------------

proc verb {} {
    if {[elec::Wrap "\\verb|" "|"]} {
	status::msg "verbatim text set"
    } else {
	status::msg "enter verbatim text"
    }
}
proc footnote {} {
    if {[elec::Wrap "\\footnote{" "}"]} {
	status::msg "footnote set"
    } else {
	status::msg "enter footnote"
    }
}
proc marginalNote {} {
    if {[elec::Wrap "\\marginpar{" "}"]} {
	status::msg "marginal note set"
    } else {
	status::msg "enter marginal note"
    }
}

proc insertLabel {} {
    if {[elec::Wrap "\\label{" "}"]} {
	status::msg "label defined"
    } else {
	status::msg "enter label"
    }
}
proc ref {} { 
    global completion::in_progress_pos
    if {[pos::compare [set completion::in_progress_pos] == [getPos]]} {
	bind::Completion
    } else {
	if {[elec::Wrap "\\ref{" "}" 1]} {
	    status::msg "reference made"
	} else {
	    status::msg "enter reference label"
	}
    }
}
proc eqref {} { 
    global completion::in_progress_pos standardTeXLabelDelimiter
    if {[pos::compare [set completion::in_progress_pos] == [getPos]]} {
	bind::Completion
    } else {
	if {[elec::Wrap "\\eqref\{eq${standardTeXLabelDelimiter}" "\}" 1]} {
	    status::msg "reference made"
	} else {
	    status::msg "enter reference label"
	}
    }
}
proc pageref {} { 
    global completion::in_progress_pos
    if {[pos::compare [set completion::in_progress_pos] == [getPos]]} {
	bind::Completion
    } else {
	if {[elec::Wrap "\\pageref{" "}" 1]} {
	    status::msg "page reference made"
	} else {
	    status::msg "enter page reference label"
	}
    }
}
proc cite {} {
    if {[elec::Wrap "\\cite{" "}" 1]} {
	status::msg "citation made"
    } else {
	status::msg "enter citation key"
    }
}
proc nocite {} {
    if {[elec::Wrap "\\nocite{" "}"]} {
	status::msg "citation added to the list"
    } else {
	status::msg "enter citation key"
    }
}

# Insert an \item or a \bibitem, depending on the context.
proc insertItem {} {
    set command [eval getText [searchEnvironment]]
    set environment [extractCommandArg $command]
    switch $environment {
	"itemize" {
	    set text "\\item  "
	}
	"enumerate" {
	    set text "\\item  "
	}
	"description" {
	    set text "\\item\[\]  "
	}
	"thebibliography" {
	    set text "\\bibitem{}  "
	}
	default {
	    beep
	    status::msg "insertItem: cursor in $environment environment"
	    return
	}
    }
    set pos [getPos]
    # Indentation should mirror that of an existing \item
    # (if it exists)
    elec::Insertion [openingCarriageReturn]$text
}

proc quotes {} {
    if {[elec::Wrap "`" "'"]} {
	status::msg "text quoted"
    } else {
	status::msg "enter text"
    }
}
proc dblQuotes {} {
    if {[elec::Wrap "``" "''"]} {
	status::msg "text double quoted"
    } else {
	status::msg "enter text"
    }
}

proc texLogo       {} {insertObject "\\TeX"}
proc latexLogo     {} {insertObject "\\LaTeX"}
proc latex2eLogo   {} {insertObject "\\LaTeXe"}
proc today         {} {insertObject "\\today"}

proc dag           {} {insertObject "\\dag"}
proc ddag          {} {insertObject "\\ddag"}
proc sectionMark   {} {insertObject "\\S"}
proc paragraphMark {} {insertObject "\\P"}
proc copyright     {} {insertObject "\\copyright"}
proc pounds        {} {insertObject "\\pounds"}


#############################################################################
# 
# Math Mode Macros
#
#############################################################################

#--------------------------------------------------------------------------
#  Math Modes: 
#--------------------------------------------------------------------------

proc texMath {} {
    checkMathMode "texMath" 0
    if {[elec::Wrap "$" "$"]} {
	status::msg "formula set"
    } else {
	status::msg "enter formula"
    }
}
proc texDisplaymath {} {
    checkMathMode "texDisplaymath" 0
    if {[elec::Wrap "$$" "$$"]} {
	status::msg "displayed formula set"
    } else {
	status::msg "enter displayed formula"
    }
}
proc latexMath {} {
    checkMathMode "latexMath" 0
    if {[elec::Wrap "\\( " " \\)"]} {
	status::msg "formula set"
    } else {
	status::msg "enter formula"
    }
}
proc latexDisplaymath {} {
    checkMathMode "latexDisplaymath" 0
    if {[elec::Wrap "\\\[ " " \\\]"]} {
	status::msg "displayed formula set"
    } else {
	status::msg "enter displayed formula"
    }
}

#--------------------------------------------------------------------------
#  Math Style: 
#--------------------------------------------------------------------------

proc doMathStyle {mathStyle description} {
    checkMathMode "$mathStyle" 1
    if {[elec::Wrap "\\$mathStyle{" "}"]} {
	status::msg "selected text is $description"
    } else {
	status::msg "enter text to be $description"
    }
}
proc doUppercaseMathStyle {mathStyle description} {
    checkMathMode "$mathStyle" 1
    # Allow upper-case alphabetic arguments only:
    if {[isSelection] && (![isUppercase] || ![isAlphabetic])} {
	beep
	alertnote "argument to \\$mathStyle must be UPPERCASE alphabetic"
	return
    }
    if {[elec::Wrap "\\$mathStyle{" "}"]} {
	status::msg "selected text is $description"
    } else {
	status::msg "enter text to be $description (UPPERCASE letters only)"
    }
}


#--------------------------------------------------------------------------
#  Formulas: 
#--------------------------------------------------------------------------

proc subscript {} {
    checkMathMode "subscript" 1
    if {[elec::Wrap "_{" "}"]} {
	status::msg "subscript set"
    } else {
	status::msg "enter subscript"
    }
}
proc superscript {} {
    checkMathMode "superscript" 1
    if {[elec::Wrap "^{" "}"]} {
	status::msg "superscript set"
    } else {
	status::msg "enter superscript"
    }
}
proc fraction {} {
    checkMathMode "fraction" 1
    set currentPos [getPos]
    if {[isSelection]} {
	set selection [getSelect]
	set args [split $selection /]
	set len [llength $args]
	deleteText $currentPos [selEnd]
	if {$len == 1} {
	    insertObject "\\frac{$selection}{}"
	    status::msg "enter denominator"
	} else {
	    set firstArg [lindex $args 0]
	    set restArgs [lrange $args 1 [expr $len-1]]
	    insertObject "\\frac{$firstArg}{[join $restArgs /]}"
	    if {$len > 2} {status::msg "beware of multiple /"}
	}
    } else {
	insertObject "\\frac{}{}"
	status::msg "enter numerator"
    }
}
proc squareRoot {} {
    checkMathMode "squareRoot" 1
    if {[elec::Wrap "\\sqrt{" "}"]} {
	status::msg "square root set"
    } else {
	status::msg "enter formula"
    }
}
proc nthRoot {} {
    checkMathMode "nthRoot" 1
    if {[elec::Wrap "\\sqrt\[\]{" "}"]} {
	status::msg "enter root"
    } else {
	status::msg "enter root, then formula"
    }
}
proc oneParameter {} {
    checkMathMode "oneParameter" 1
    if {[elec::Wrap "\\{" "}"]} {
	status::msg "enter command name"
    } else {
	status::msg "enter command name, press <Tab>, enter argument"
    }
}
proc twoParameters {} {
    checkMathMode "twoParameters" 1
    if {[elec::Wrap "\\{" "}{}"]} {
	status::msg "enter command name"
    } else {
	status::msg "enter command name, press <Tab>, enter argument, etc."
    }
}

#--------------------------------------------------------------------------
#  Large Ops: 
#--------------------------------------------------------------------------

proc insertLargeOp {commandName} {
    checkMathMode "$commandName" 1
    insertObject "\\$commandName\_{}^{}"
}
proc sum       {} {insertLargeOp "sum"}
proc prod      {} {insertLargeOp "prod"}
proc coprod    {} {insertLargeOp "coprod"}
proc int       {} {insertLargeOp "int"}
proc oint      {} {insertLargeOp "oint"}
proc bigcap    {} {insertLargeOp "bigcap"}
proc bigcup    {} {insertLargeOp "bigcup"}
proc bigsqcup  {} {insertLargeOp "bigsqcup"}
proc bigvee    {} {insertLargeOp "bigvee"}
proc bigwedge  {} {insertLargeOp "bigwedge"}
proc bigodot   {} {insertLargeOp "bigodot"}
proc bigotimes {} {insertLargeOp "bigotimes"}
proc bigoplus  {} {insertLargeOp "bigoplus"}
proc biguplus  {} {insertLargeOp "biguplus"}

#--------------------------------------------------------------------------
#  Delimiters: 
#--------------------------------------------------------------------------

proc delimitObject {leftDelim rightDelim} {
    if {[elec::Wrap $leftDelim $rightDelim]} {
	status::msg "formula delimited"
    } else {
	status::msg "enter formula"
    }
}
proc parentheses   {} {checkMathMode "parentheses" 1;   delimitObject "(" ")"}
proc brackets      {} {checkMathMode "brackets" 1;      delimitObject "\[" "\]"}
proc braces        {} {checkMathMode "braces" 1;        delimitObject "\\\{" "\\\}"}
proc absoluteValue {} {checkMathMode "absoluteValue" 1; delimitObject "|" "|"}
proc getDelims     {} {
    catch {prompt "Choose delimiters:" "parentheses" "" "parentheses" \
      "brackets" "braces" "angle brackets" "vertical bars" \
      "double bars" "ceiling" "floor"} delimType
    if {$delimType != "cancel"} {
	switch $delimType {
	    "parentheses" {
		set leftDelim "("
		set rightDelim ")"
	    }
	    "brackets" {
		set leftDelim "\["
		set rightDelim "\]"
	    }
	    "braces" {
		set leftDelim "\\\{"
		set rightDelim "\\\}"
	    }
	    "vertical bars" {
		set leftDelim "|"
		set rightDelim "|"
	    }
	    "double bars" {
		set leftDelim "\\|"
		set rightDelim "\\|"
	    }
	    "angle brackets" {
		set leftDelim "\\langle"
		set rightDelim "\\rangle"
	    }
	    "ceiling" {
		set leftDelim "\\lceil"
		set rightDelim "\\rceil"
	    }
	    "floor" {
		set leftDelim "\\lfloor"
		set rightDelim "\\rfloor"
	    }
	    default {
		alertnote "\"$delimType\" not recognized"
		return ""
	    }
	}
	return [list $leftDelim $rightDelim]
    } else {return ""}
}
proc otherDelims {} {
    checkMathMode "otherDelims" 1
    set delims [getDelims]
    if {$delims != ""} {
	set leftDelim [lindex $delims 0]
	set rightDelim [lindex $delims 1]
	delimitObject "$leftDelim" "$rightDelim"
    }
}
proc {half-openInterval} {} {
    checkMathMode "half-openInterval" 1; delimitObject "(" "\]"
}
proc {half-closedInterval} {} {
    checkMathMode "half-closedInterval" 1; delimitObject "\[" ")"
}
proc insertBigDelims {leftDelim rightDelim isMultiline} {
    checkMathMode "insertBigDelims" 1
    if {$isMultiline} {
	doWrapStructure $leftDelim "" $rightDelim
    } else {
	if {[elec::Wrap $leftDelim $rightDelim]} {
	    status::msg "formula delimited"
	} else {
	    status::msg "enter formula"
	}
    }
}
proc bigParens {} {
    checkMathMode "bigParens" 1; insertBigDelims "\\left(" "\\right)" 0
}
proc multiBigParens {} {
    checkMathMode "multiBigParens" 1; insertBigDelims "\\left(" "\\right)" 1
}
proc bigBrackets {} {
    checkMathMode "bigBrackets" 1; insertBigDelims "\\left\[" "\\right\]" 0
}
proc multiBigBrackets {} {
    checkMathMode "multiBigBrackets" 1; insertBigDelims "\\left\[" "\\right\]" 1
}
proc bigBraces {} {
    checkMathMode "bigBraces" 1; insertBigDelims "\\left\\\{" "\\right\\\}" 0
}
proc multiBigBraces {} {
    checkMathMode "multiBigBraces" 1; insertBigDelims "\\left\\\{" "\\right\\\}" 1
}
proc bigAbsValue {} {
    checkMathMode "bigAbsValue" 1; insertBigDelims "\\left|" "\\right|" 0
}
proc multiBigAbsValue {} {
    checkMathMode "multiBigAbsValue" 1; insertBigDelims "\\left|" "\\right|" 1
}
proc doOtherBigDelims {name isMultiline} {
    checkMathMode $name 1
    set delims [getDelims]
    if {$delims != ""} {
	append leftDelim "\\left" [lindex $delims 0]
	append rightDelim "\\right" [lindex $delims 1]
	insertBigDelims "$leftDelim" "$rightDelim" $isMultiline
    }
}
proc otherBigDelims {} {
    doOtherBigDelims "otherBigDelims" 0
}
proc otherMultiBigDelims {} {
    doOtherBigDelims "otherMultiBigDelims" 1
}
proc bigLeftBrace {} {
    checkMathMode "bigLeftBrace" 1
    insertBigDelims "\\left\\\{" "\\right." 0
}
proc multiBigLeftBrace {} {
    checkMathMode "multiBigLeftBrace" 1
    insertBigDelims "\\left\\\{" "\\right." 1
}
proc doOtherMixedBigDelims {name isMultiline} {
    checkMathMode $name 1
    catch {prompt "Choose LEFT delimiter:" "parenthesis" "" "parenthesis" \
      "bracket" "brace" "vertical bar" "double bar" \
      "angle bracket" "ceiling" "floor" "slash" "backslash" \
      "none"} delimType
    if {$delimType != "cancel"} {
	switch $delimType {
	    "parenthesis" {set leftDelim "("}
	    "bracket" {set leftDelim "\["}
	    "brace" {set leftDelim "\\\{"}
	    "vertical bar" {set leftDelim "|"}
	    "double bar" {set leftDelim "\\|"}
	    "angle bracket" {set leftDelim "\\langle"}
	    "ceiling" {set leftDelim "\\lceil"}
	    "floor" {set leftDelim "\\lfloor"}
	    "slash" {set leftDelim "/"}
	    "backslash" {set leftDelim "\\backslash"}
	    "none" {set leftDelim "."}
	    default {
		alertnote "\"$delimType\" not recognized"
		return
	    }
	}
	catch {prompt "Choose RIGHT delimiter:" "parenthesis" "" "parenthesis" \
	  "bracket" "brace" "vertical bar" "double bar" \
	  "angle bracket" "ceiling" "floor" "slash" "backslash" \
	  "none"} delimType
	if {$delimType != "cancel"} {
	    switch $delimType {
		"parenthesis" {set rightDelim ")"}
		"bracket" {set rightDelim "\]"}
		"brace" {set rightDelim "\\\}"}
		"vertical bar" {set rightDelim "|"}
		"double bar" {set rightDelim "\\|"}
		"angle bracket" {set rightDelim "\\rangle"}
		"ceiling" {set rightDelim "\\rceil"}
		"floor" {set rightDelim "\\rfloor"}
		"slash" {set rightDelim "/"}
		"backslash" {set rightDelim "\\backslash"}
		"none" {set rightDelim "."}
		default {
		    alertnote "\"$delimType\" not recognized"
		    return
		}
	    }
	    insertBigDelims "\\left$leftDelim" "\\right$rightDelim" $isMultiline
	}
    }
}
proc otherMixedBigDelims {} {
    doOtherMixedBigDelims "otherMixedBigDelims" 0
}
proc otherMultiMixedBigDelims {} {
    doOtherMixedBigDelims "otherMultiMixedBigDelims" 1
}

#--------------------------------------------------------------------------
#  Accents: 
#--------------------------------------------------------------------------

proc acute {} {
    checkMathMode "acute" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\acute{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc bar {} {
    checkMathMode "bar" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\bar{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc breve {} {
    checkMathMode "breve" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\breve{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc check {} {
    checkMathMode "check" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\check{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc dot {} {
    checkMathMode "dot" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\dot{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc ddot {} {
    checkMathMode "ddot" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\ddot{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc grave {} {
    checkMathMode "grave" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\grave{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc hat {} {
    checkMathMode "hat" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\hat{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc tilde {} {
    checkMathMode "tilde" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\tilde{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc vec {} {
    checkMathMode "vec" 1
    if {[isSelection] > 1} {
	alertnote "Warning: only a single character may be accented!"
    }
    if {[elec::Wrap "\\vec{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter one character"
    }
}
proc widehat {} {
    checkMathMode "widehat" 1
    if {[isSelection] > 3} {
	alertnote "Warning: only a few characters may be accented!"
    }
    if {[elec::Wrap "\\widehat{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter a few characters"
    }
}
proc widetilde {} {
    checkMathMode "widetilde" 1
    if {[isSelection] > 3} {
	alertnote "Warning: only a few characters may be accented!"
    }
    if {[elec::Wrap "\\widetilde{" "}"]} {
	status::msg "accent set"
    } else {
	status::msg "enter a few characters"
    }
}

#--------------------------------------------------------------------------
#  Grouping: 
#--------------------------------------------------------------------------
proc underline {} {
    checkMathMode "underline" 1
    if {[elec::Wrap "\\underline{" "}"]} {
	status::msg "selection underlined"
    } else {
	status::msg "enter text"
    }
}
proc overline {} {
    checkMathMode "overline" 1
    if {[elec::Wrap "\\overline{" "}"]} {
	status::msg "selection overlined"
    } else {
	status::msg "enter text"
    }
}
proc underbrace {} {
    checkMathMode "underbrace" 1
    if {[elec::Wrap "\\underbrace{" "}"]} {
	status::msg "selection underbraced"
    } else {
	status::msg "enter text"
    }
}
proc overbrace {} {
    checkMathMode "overbrace" 1
    if {[elec::Wrap "\\overbrace{" "}"]} {
	status::msg "selection overbraced"
    } else {
	status::msg "enter text"
    }
}
proc overrightarrow {} {
    checkMathMode "overrightarrow" 1
    if {[elec::Wrap "\\overrightarrow{" "}"]} {
	status::msg "selection overrightarrowed"
    } else {
	status::msg "enter text"
    }
}
proc overleftarrow {} {
    checkMathMode "overleftarrow" 1
    if {[elec::Wrap "\\overleftarrow{" "}"]} {
	status::msg "selection overleftarrowed"
    } else {
	status::msg "enter text"
    }
}
proc stackrel {} {
    checkMathMode "stackrel" 1
    set currentPos [getPos]
    if {[insertObject "\\stackrel{}{}"]} {
	status::msg "1st arg scriptstyle"
    }
}

#--------------------------------------------------------------------------
#  Spacing: 
#--------------------------------------------------------------------------

proc negThin {} {checkMathMode "negThin" 1; insertObject "\\!"}
proc thin    {} {checkMathMode "thin" 1;    insertObject "\\,"}
proc medium  {} {checkMathMode "medium" 1;  insertObject "\\:"}
proc thick   {} {checkMathMode "thick" 1;   insertObject "\\;"}
proc quad    {} {checkMathMode "quad" 1;    insertObject "\\quad"}
proc qquad   {} {checkMathMode "qquad" 1;   insertObject "\\qquad"}
proc hspace  {} {
    checkMathMode "hspace" 1
    if {[elec::Wrap "\\hspace{" "}"]} {
	status::msg "spacing set"
    } else {
	status::msg "enter the desired horizontal spacing"
    }
}
proc vspace {} {
    checkMathMode "vspace" 1
    if {[elec::Wrap "\\vspace{" "}"]} {
	status::msg "spacing set"
    } else {
	status::msg "enter the desired horizontal spacing"
    }
}
proc hfill     {} {checkMathMode "hfill" 1;     insertObject "\\hfill"}
proc vfill     {} {checkMathMode "vfill" 1;     insertObject "\\vfill"}
proc smallskip {} {checkMathMode "smallskip" 1; insertObject "\\smallskip"}
proc medskip   {} {checkMathMode "medskip" 1;   insertObject "\\medskip"}
proc bigskip   {} {checkMathMode "bigskip" 1;   insertObject "\\bigskip"}
